package choco;

import org.chocosolver.solver.Model;
import org.chocosolver.solver.variables.IntVar;
import org.chocosolver.util.tools.ArrayUtils;
import org.chocosolver.solver.Solver;
import static org.chocosolver.solver.search.strategy.Search.*;

public class golfers {
	
	public static void main(String[] args) {
		int nw = 3; // 2, 4, 3 - number of weeks
		int ng = 8; // 4, 4, 8 - number of groups		
		int gs = 4; // 3, 3, 4 - groups size
		int n = ng*gs; // number of golfers
		Model model = new Model("Social Golfers (" + nw + "," + ng + ","+ gs + ")");
		Solver s = model.getSolver();
		
		boolean first_fail = false;
		
		IntVar[][] groups = model.intVarMatrix("G", nw, n, 1, ng);
				
		
		// in every week, golfer 0 always play in group 1  (symmetry breaking constraint)
		for (int w = 0; w < nw; w++){
			model.arithm(groups[w][0], "=", 1).post();
		}
		
		/* 2, 4, 3
		   number of failures = 	462		142		53
     		number of backtracks = 	921		281		105
    		elapsed cpu time (ms) = 134		75		52
		 */
		
		// the first week is arbitrarily fixed (symmetry breaking constraint)
		for (int p = 0; p < n; p++){
			int gg = p/gs + 1;
			//System.out.println(p + "->" + gg);
			model.arithm(groups[0][p], "=", gg).post();
		}
		
		// the weeks are in increasing order (symmetry breaking constraint)
		for (int w1 = 0; w1 < nw-1 ; w1++){
			for (int w2 = w1+1; w2 < nw ; w2++){
				model.arithm(groups[w1][1], "<", groups[w2][1]).post();
			}
		}
		//Create an IntVar with the value of gs (for the next constraint)
		IntVar gsv = model.intVar(gs,gs);
		
		// in every week, each group is assigned to exactly gs golfers
		for (int w = 0; w < nw; w ++){
			for (int g = 1; g <= ng; g ++){
				model.count(g, groups[w], gsv).post();
			}
		}
		
		// no two golfers meet in different weeks
		for(int w1=0; w1 < nw-1; w1++){
			for(int w2=w1+1; w2 < nw; w2++){
				for(int g1=0; g1 < n-1; g1++){
					for(int g2=g1+1; g2 < n; g2++){						
						model.or(
							model.arithm(groups[w1][g1], "!=", groups[w1][g2]),
							model.arithm(groups[w2][g1], "!=", groups[w2][g2]))
						.post();						
					}
				}
			}	
		}
		
		// first-fail search on flattened flat variables
		if (first_fail){
			IntVar[] flat = {};
			for (int w = 0; w < nw; w++){
				for (int p = 0; p < n; p++){
					flat = ArrayUtils.concat(flat,groups[w][p]);
				}
			}
			s.setSearch(minDomLBSearch(flat));
		}

		if(s.solve()){
			show_weeks(groups, nw, n);
			show_solution(groups, nw, ng, gs);
			// show_clashes(groups, nw, ng, gs);
		} else {
			System.out.print("no solution");
		}
		System.out.println("\n");
		//s.printStatistics();
		System.out.println("       number of failures = " + String.valueOf(s.getFailCount()));
		System.out.println("     number of backtracks = " + String.valueOf(s.getBackTrackCount()));
		System.out.println("    elapsed cpu time (ms) = " + String.valueOf(s.getTimeCount()*1000));
		//s.printStatistics();
	}
	
	public static void show_weeks(IntVar[][] groups, int nw, int n){
		for(int w = 0; w < nw; w++){
			System.out.println();
			for(int j = 0; j < n; j++){
				System.out.print(" " + groups[w][j].getValue());
			}
		}
		System.out.println();
	}
	
	public static void show_solution(IntVar[][] groups, int nw, int ng, int gs){
		for(int w = 0; w < nw; w++){
			System.out.println();
			for(int g = 1; g <= ng; g++){
				for (int p = 0; p < ng*gs; p++){
					if (groups[w][p].getValue() == g){
						System.out.print(" "+ p);
					}
				}
				System.out.print(" | ");
			}
		}
	}
	
	public static void show_clashes(IntVar[][] groups, int nw, int ng, int gs){
		System.out.println();
		System.out.println();
		for(int g1 = 0; g1 < ng*gs-1; g1++){
			for(int g2 = g1+1; g2 < ng*gs; g2++){
				for(int w1 = 0; w1 < nw-1; w1++){
					for(int w2 = w1+1; w2 < nw; w2++){
					if ((groups[w1][g1].getValue() == groups[w1][g2].getValue()) &&
						(groups[w2][g1].getValue() == groups[w2][g2].getValue()))
						System.out.println(g1 + " and " + g2 + " meet at weeks " + w1 + " and " + w2);
					}
				}
			}
		}
		System.out.println( " no (more) clashes");
	}


}