package choco;

import org.chocosolver.solver.Model;
import org.chocosolver.solver.variables.IntVar;
import org.chocosolver.solver.*;

/*
 The problem is to put n balls labelled 1,…,n into 3 boxes so that for any triple of balls
 (x,y,z), with x+y=z, not all are in the same box. This has a solution iff n < 14. 
 
 One natural generalization is to consider partitioning into k boxes (for k>3).
 
 3 2 2 3 1 ...
 
 [(box(x) == box(y) )] => (box(x+y) != box(y))
 [(box(x) != box(y) )] \/ (box(x+y) != box(y))
 
*/
public class schurs_lemma {
	
	public static void main(String[] args) {
		int n_boxes = 4;  // b =  3 / 4
		int n_balls = 40; // n = 13 /40 (max n = 13 for b = 3)

		Model m = new Model("(" + n_boxes + "," + n_balls + ") - schur's lemma");
		
 		// Ball i is in Box j iff box[i-1] = j 
		IntVar[] box = m.intVarArray("B", n_balls, 1, n_boxes);

		// Ball i is in the same box of Ball j then the box of Ball i+j is not the same
		for(int x = 1; x <= n_balls; x++){
			for (int y = x; y <= n_balls; y++) {
				if (x+y <= n_balls){
					m.or(	m.arithm(box[x-1], "!=", box[y-1]), 
							m.arithm(box[x-1], "!=", box[x+y-1]))
					.post();
				}
			}
		}
		
 		Solver s = m.getSolver();
 		float t1 = s.getTimeCount()*1000;
 		Solution solution = s.findSolution();		
		float t2 = s.getTimeCount()*1000;
		if(solution != null){
			System.out.println("\n");
			m.getSolver().printStatistics();
			System.out.println(" --- Statistics ");
			System.out.println("       number of failures = " + String.valueOf(s.getFailCount()));
			System.out.println("     number of backtracks = " + String.valueOf(s.getBackTrackCount()));
			System.out.println("    elapsed cpu time (ms) = " + String.valueOf(s.getTimeCount()*1000));
			System.out.println();
			for (int b = 1; b <= n_boxes; b++){
				System.out.print("box " + b + ":");
				for (int i = 0; i < n_balls; i++){
					if (box[i].getValue() == b){
						System.out.print(" " + (i+1));
					}
				}
				System.out.print("\n");

			}
			System.out.print((t2-t1) + "\n");	
			
		} else {
			System.out.print("no solutions\n");
			m.getSolver().printStatistics();
		}
		
	}
}